/*
 *  PerlinNoise.java
 */

package Experiment;

// two-dimensional grid of fractal-esque noise values
public class PerlinNoise {
  
  // an array of noise values between zero and one
  public float[][] mData = null;
  
  // constructor
  // noise is periodic (works best if grid is n x n with n a power of 2) 
  // persist is around 0.5: bigger means low frequencies dominate
  //                        smaller means high frequencies dominate
  public PerlinNoise(int xSize, int ySize, float persist) {
    
    mData = new float[xSize][ySize];
    for ( int i = 0 ; i < xSize ; i++ ) {
      for ( int j = 0 ; j < ySize ; j++ ) {
        mData[i][j] = 0.0f;
      }
    }
    
    // number of layers of noise to apply
    int numLayers = (int)Math.floor(Math.log(Math.max(xSize, ySize))
                                   /Math.log(2.0)) - 2;

    // build up layers of noise
    for ( int layer = 0 ; layer < numLayers ; layer++ ) {

      // make a square array of random values
      int step = (int)Math.pow(2.0, layer),
          xNum = Math.max(1, xSize/step),
          yNum = Math.max(1, ySize/step);
      float[][] rand = new float[xNum][yNum];
      for ( int i = 0 ; i < xNum ; i++ ) {
        for ( int j = 0 ; j < yNum ; j++ ) {
          rand[i][j] = Env.randomFloat();
        }
      }
      
      // amplitude of layer decreases with frequency of noise
      float amp = (float)Math.pow(1.0f-persist, -layer);
      
      // interpolate the random values onto the grid
      // first interpolate along the y-direction
      float[][] yInterp = new float[xNum][ySize];
      for ( int i = 0 ; i < xNum ; i++ ) {
        interpolate(rand[i], yInterp[i]);
      }
      // now interpolate along the x-direction
      for ( int j = 0 ; j < ySize ; j++ ) {
        float[] xVals = new float[xNum];
        for ( int i = 0 ; i < xNum ; i++ ) xVals[i] = yInterp[i][j];
        float[] xInterp = new float[xSize];
        interpolate(xVals, xInterp);
        for ( int i = 0 ; i < xSize ; i++ ) mData[i][j] += amp*xInterp[i];
      }
      
    } // for each layer
    
    // normalize the result to between zero and one
    float minVal = mData[0][0],
          maxVal = mData[0][0];
    for ( int i = 0 ; i < xSize ; i++ ) {
      for ( int j = 0 ; j < ySize ; j++ ) {
        minVal = Math.min(minVal, mData[i][j]);
        maxVal = Math.max(maxVal, mData[i][j]);
      }
    }
    for ( int i = 0 ; i < xSize ; i++ ) {
      for ( int j = 0 ; j < ySize ; j++ ) {
        mData[i][j] = (mData[i][j] - minVal)/(maxVal - minVal);
      }
    }
    
  } // constructor
  
  // one-dimensional interpolation, selects between first- and third-order
  protected void interpolate(float[] source, float[] dest) {
    
    if ( dest.length <= 2*source.length ) interpolateLinear(source, dest);
    else                                  interpolateCubic(source, dest);
    
  } // interpolate()
  
  // piecewise first-order interpolation
  protected void interpolateLinear(float[] source, float[] dest) {
    
    for ( int i = 0 ; i < dest.length ; i++ ) {
      float x  = (i/(float)dest.length)*source.length;
      int   i0 = (int)Math.floor(x);
      float x0 = x - i0;
      float v0 = source[i0],
            v1 = source[Env.fold(i0+1,source.length)];
      dest[i]  = v0 + (v1-v0)*x0;
    }
    
  } // interpolateLinear
  
  // smooth piecewise third-order interpolation
  protected void interpolateCubic(float[] source, float[] dest) {
    
    // estimate gradient at each source point
    float[] grads = new float[source.length];
    for ( int i = 0 ; i < source.length ; i++ ) {
      grads[i] = 0.5f*( source[Env.fold(i+1,source.length)] 
                      - source[Env.fold(i-1,source.length)] );
    }
    
    // interpolate
    for ( int i = 0 ; i < dest.length ; i++ ) {
      float x  = (i/(float)dest.length)*source.length;
      int   i0 = (int)Math.floor(x);
      float x0 = x - i0;
      float v0 = source[i0],
            v1 = source[Env.fold(i0+1,source.length)],
            g0 = grads[i0],
            g1 = grads[Env.fold(i0+1,source.length)];
      dest[i]  = (((g0+g1-2*v1+2*v0)*x0 + (3*v1-3*v0-2*g0-g1))*x0 + g0)*x0 + v0;
    }
    
  } // interpolateCubic
  
} // class PerlinNoise
