/*
 *  Experiment.java
 */

package Experiment;

import java.awt.*;
import java.awt.event.*;
import java.util.*;

import javax.media.opengl.*;
import javax.media.opengl.glu.*;
import com.sun.opengl.util.*;

// the frame for the game
public class MainWindow extends Frame implements GLEventListener {

  // main
  public static void main(String[] args) {

    //!!!
    Env.setVerbose();
    //!!!
    
    boolean unknownArg = false;
    for ( int k = 0 ; k < args.length ; k++ ) {
      String arg = args[k].toLowerCase();
      if   ( arg.equals("debug") ) Env.setVerbose();
      else                         unknownArg = true;
    }
    if ( unknownArg ) {
      System.out.println("java -jar <file>.jar [game] [options]");
      System.out.println("options: debug");
      System.exit(0);
    }

    System.setProperty("sun.java2d.noddraw","true");
    
    new MainWindow();
    
  } // main()

  // this object collects timing statistics as the game is running
  private class TimingStats {
    private int  mNumTicks;
    private long mNumNanos, mNumNanosInAdvance, mNumNanosInDraw;
    private long mPeakNanos, mPeakNanosInAdvance, mPeakNanosInDraw;
    public TimingStats() {
      clear(); 
    }
    public void clear() {
      mNumTicks = 0;
      mNumNanos = mNumNanosInAdvance = mNumNanosInDraw = 0;
      mPeakNanos = mPeakNanosInAdvance = mPeakNanosInDraw = 0;
    }
    public void update(long nanosInTick,
                       long nanosInAdvance,
                       long nanosInDraw) {
      mNumTicks++;
      mNumNanos += nanosInTick;
      mPeakNanos = Math.max(nanosInTick, mPeakNanos);
      mNumNanosInAdvance += nanosInAdvance;
      mPeakNanosInAdvance = Math.max(nanosInAdvance, mPeakNanosInAdvance);
      mNumNanosInDraw += nanosInDraw;
      mPeakNanosInDraw = Math.max(nanosInDraw, mPeakNanosInDraw);
    }
    public String toString() {
      int  numTicks = Math.max(1, mNumTicks);
      long numNanos = Math.max(1, mNumNanos);
      return new String(mNumTicks
                        + " frames, "
                        + String.format("%.1f", mNumTicks/(numNanos*1.0e-9f))
                        + " per sec (mean "
                        + String.format("%.1f", mNumNanos*1.0e-6f/numTicks)
                        + "ms, max "
                        + String.format("%.1f", mPeakNanos*1.0e-6f)
                        + "ms), "
                        + (100*mNumNanosInAdvance)/numNanos
                        + "% in advance (max "
                        + String.format("%.1f", mPeakNanosInAdvance*1.0e-6f)
                        + "ms), "
                        + (100*mNumNanosInDraw)/numNanos
                        + "% in draw (max "
                        + String.format("%.1f", mPeakNanosInDraw*1.0e-6f)
                        + "ms)");
    }
  } // class TimingStats

  // maximum time step (seconds)
  private static final float kMaxTimeStep = 0.1f;
  
  // previous time of advance (nanoseconds)
  private long mPrevNanos;
  
  // next time to display statistics (nanoseconds)
  private long mNextUpdateTime;
  
  // timing statistics
  private TimingStats mTotalStats = null,
                      mLocalStats = null;
  
  // viewport aspect ratio (width/height)
  private float mAspectRatio;

  // main game object
  private Game mGame;
  
  // the JOGL animation object
  private Animator mAnimator;
  
  // constructor
  public MainWindow() {
  
    Env.initialize(this);
    
    mPrevNanos = -1;
    mNextUpdateTime = -1;
    mAspectRatio = 1.0f;
    
    mGame = new Game();
  
    GLCanvas canvas = new GLCanvas();
    canvas.addGLEventListener(this);
    add(canvas);
    Env.keys().monitor(canvas);
    
    mAnimator = new Animator(canvas); //new FPSAnimator(canvas, 60);

    int nx = 400, ny = 300;
    setSize(nx, ny);

    Dimension screenDim = Toolkit.getDefaultToolkit().getScreenSize();
    setLocation(screenDim.width/2 - nx/2, screenDim.height/2 - ny/2);
    
    setTitle("Dancing Tanks (dishmoth@yahoo.co.uk)");
    
    addWindowListener(
      new WindowAdapter() {
        public void windowClosing(WindowEvent e) { exit(); }
        public void windowDeiconified(WindowEvent e) { start(); }
        public void windowIconified(WindowEvent e) { stop(); }
      }
    );
    
    addMouseListener(
      new MouseAdapter() {
        public void mouseClicked(MouseEvent e) { requestFocusInWindow(); }
      }
    );
    canvas.addMouseListener(
      new MouseAdapter() {
        public void mouseClicked(MouseEvent e) { requestFocusInWindow(); }
      }
    );
    requestFocusInWindow();

    setVisible(true);
    
    mTotalStats = new TimingStats();
    mLocalStats = new TimingStats();

    mAnimator.start();

  } // constructor
  
  // start the game loop
  public void start() {

    requestFocusInWindow();

  } // start()
  
  // interrupt the game
  public void stop() {
    
    Env.keys().reset();
    //Storage.flushCache();
    if ( Env.performanceStats() ) {
      System.out.println("Overall: " + mTotalStats.toString());
    }
    
  } // stop()

  // end the game
  public void exit() {
 
    stop();
    // make sure the call to Animator.stop() completes before exiting
    new Thread(
      new Runnable() {
        public void run() { 
          mAnimator.stop(); 
          System.exit(0); }
      }
    ).start();
    
  } // exit()
  
  // GL initialize
  public void init(GLAutoDrawable drawable) {
    
    if ( Env.debugMode() ) {
      drawable.setGL(new DebugGL(drawable.getGL()));
      //drawable.setGL(new TraceGL(drawable.getGL(), System.out));
    }

    GL gl = drawable.getGL();

    if ( Env.debugMode() ) {
      System.out.println("GL class: " + gl.getClass().getName());
      System.out.println("GLCapabilities: " 
                         + drawable.getChosenGLCapabilities());
      System.out.println("GL_VENDOR: " + gl.glGetString(GL.GL_VENDOR));
      System.out.println("GL_RENDERER: " + gl.glGetString(GL.GL_RENDERER));
      System.out.println("GL_VERSION: " + gl.glGetString(GL.GL_VERSION));
    }

    gl.setSwapInterval(1);

    Env.setGlu(new GLU());

    mGame.init(gl);
    
  } // GLEventListener.init()

  // GL change of window size
  public void reshape(GLAutoDrawable drawable, 
                      int x, int y, int width, int height) {
    
    GL gl = drawable.getGL();

    gl.glViewport(0, 0, width, height);
    Env.setAspectRatio(width/(float)height);
    
    gl.glMatrixMode(GL.GL_PROJECTION);
    gl.glLoadIdentity();
    mGame.configureCamera(gl);
    
    gl.glMatrixMode(GL.GL_MODELVIEW);
    gl.glLoadIdentity();

  } // GLEventListener.reshape()

  // GL change of screen mode
  public void displayChanged(GLAutoDrawable drawable, 
                             boolean modeChanged, 
                             boolean deviceChanged) {}

  // main GL display routine
  public void display(GLAutoDrawable drawable) {
  
    long nanosNewTick = System.nanoTime();    
    boolean firstTime = ( mPrevNanos < 0 || mNextUpdateTime < 0 );
  
    // advance
    if ( !firstTime ) {
      float dt = Math.min(kMaxTimeStep, (nanosNewTick-mPrevNanos)*1.0e-9f);
      assert( dt >= 0.0f );
      mGame.advance(dt);
    }
  
    long nanosAfterAdvance = System.nanoTime();
    
    // display
    GL gl = drawable.getGL();
    mGame.display(gl);
    gl.glFlush();

    long nanosAfterDraw = System.nanoTime();

    // update timing statistics
    if ( firstTime ) {
      mNextUpdateTime = nanosNewTick + 1000000000L;
    } else {    
      updateProgress(nanosNewTick - mPrevNanos,
                     nanosAfterAdvance - nanosNewTick,
                     nanosAfterDraw - nanosAfterAdvance);
      if ( Env.performanceStats() && nanosNewTick > mNextUpdateTime ) {
        System.out.println(mLocalStats.toString());
        mLocalStats.clear();
        mNextUpdateTime = nanosNewTick + 1000000000L;
      }
    }
    mPrevNanos = nanosNewTick;

  } // GLEventListener.display()

  // update the running statistics and the overall statistics
  private void updateProgress(long    nanosInTick,
                              long    nanosInAdvance,
                              long    nanosInDraw) {
    
    mTotalStats.update(nanosInTick, nanosInAdvance, nanosInDraw);
    mLocalStats.update(nanosInTick, nanosInAdvance, nanosInDraw);
    
  } // updateProgress()
  
} // class MainWindow
