/*
 *  KeyMonitor.java
 */

package Experiment;

import java.awt.*;
import java.awt.event.*;

// keep track of what keys are currently pressed
public class KeyMonitor implements KeyListener {

  // number of keys codes that we pay attention to
  static private final int kNumKeyCodes = 256;

  // table of states (pressed or not) for the key codes we're listening to
  private boolean mKeyStates[];

  // whether any key pressed event has occurred
  private boolean mAnyKeyPressed;
  
  // constructor: declare as a key listener to the owner
  public KeyMonitor(Component owner) {
    
    reset();
    if ( owner != null ) owner.addKeyListener(this);
    
  } // constructor
  
  // monitor the keys on other components
  public void monitor(Component target) {
    
    if ( target != null ) target.addKeyListener(this);
    
  } // monitor()
  
  // assert that all keys are currently not pressed
  public void reset() {
    
    if ( mKeyStates == null ) mKeyStates = new boolean[kNumKeyCodes];
    for ( int i = 0 ; i < kNumKeyCodes ; i++ ) mKeyStates[i] = false;
    mAnyKeyPressed = false;
    
  } // reset()
  
  // check whether a specific key is currently pressed
  public boolean pressed(int keyCode) {
  
    if ( keyCode < 0 || keyCode >= kNumKeyCodes ) return false;
    return (mKeyStates[keyCode]);
  
  } // pressed()
  
  // check whether any of a group of keys is currently pressed
  public boolean up()    { return ( pressed(KeyEvent.VK_UP)
                                 || pressed(KeyEvent.VK_SHIFT)
                                 || pressed(KeyEvent.VK_ENTER)
                                 || pressed(KeyEvent.VK_Q) ); }
  public boolean down()  { return ( pressed(KeyEvent.VK_DOWN)
                                 || pressed(KeyEvent.VK_A) ); }
  public boolean left()  { return ( pressed(KeyEvent.VK_LEFT)
                                 || pressed(KeyEvent.VK_O) ); }
  public boolean right() { return ( pressed(KeyEvent.VK_RIGHT)
                                 || pressed(KeyEvent.VK_P) ); }
  public boolean fire()  { return ( pressed(KeyEvent.VK_CONTROL)
                                 || pressed(KeyEvent.VK_SPACE)
                                 || pressed(KeyEvent.VK_M)
                                 || pressed(KeyEvent.VK_Z) ); }
  
  // check whether any of our main keys is currently pressed
  public boolean any() {

    for ( int i = 0 ; i < kNumKeyCodes ; i++ ) {
      if ( mKeyStates[i] ) return true;
    }
    return false; 
  
  } // any()
  
  // check whether any key has been pressed since the last reset
  public boolean anySinceReset() { return mAnyKeyPressed; }
  
  // implementation of KeyListener interface
  public void keyPressed(KeyEvent e) {
    
    int keyCode = e.getKeyCode();
    if ( keyCode < 0 || keyCode >= kNumKeyCodes ) return;
    mKeyStates[keyCode] = true;
    
    mAnyKeyPressed= true;
    
  } // KeyListener.keyPressed()
  
  // implementation of KeyListener interface
  public void keyReleased(KeyEvent e) {

    int keyCode = e.getKeyCode();
    if ( keyCode < 0 || keyCode >= kNumKeyCodes ) return;
    mKeyStates[keyCode] = false;

  } // KeyListener.keyReleased()
  
  // implementation of KeyListener interface
  public void keyTyped(KeyEvent e) {
  } // KeyListener.keyTyped()

} // class KeyMonitor
