/*
 *  Game.java
 */
 
package Experiment;

import java.awt.*;
import java.awt.event.*;
import java.util.*;

import javax.media.opengl.*;
import javax.media.opengl.glu.*;
import com.sun.opengl.util.*;

// controller of the game
public class Game {

  // light details
  private static final float kLightFirstAngle   = 45.0f,
                             kLightElevations[] = { 45.0f, 30.0f, 20.0f },
                             kLightDiffuse[]    = { 1.0f, 1.0f, 1.0f, 1.0f },
                             kLightSpecular[]   = { 1.0f, 1.0f, 1.0f, 1.0f };
  
  // camera details
  private static final float kCameraY         = 1.9f,
                             kCameraLensScale = 2.0f,
                             kCameraOffset    = 5.0f,
                             kCameraNearPlane = kCameraOffset + 2.0f,
                             kCameraFarPlane  = kCameraOffset + 500.0f;

  // overview mode details
  private static final float kBackgroundColour[]   = {0.3f, 0.6f, 0.9f},
                             kOverviewColour[]     = {0.4f, 0.7f, 1.0f, 1.0f};
  private static final float kOverviewScaleInitial = 40.0f,
                             kOverviewScaleRate    = 0.5f,
                             kOverviewScaleMin     = 5.0f,
                             kOverviewScaleMax     = 1000.0f;
  
  // width of outlines
  private static final float kLineWidth = 1.2f;
  
  // details of 'reflective' surface
  private static final float kSurfaceColourNear[] = { 0.3f, 0.6f, 0.9f, 0.5f },
                             kSurfaceColourFar[]  = { 0.6f, 0.6f, 0.7f, 0.95f };
  private static final float kSurfaceNearMark     = -0.95f,
                             kSurfaceFarMark      = -0.15f;
  
  // directions to the light sources
  private float mLightDirections[][] = null;

  // show scene from above
  private boolean mOverviewMode,
                  mOverviewToggle;
  private float   mOverviewScale;
  
  // current camera details
  private float mCameraX,
                mCameraZ,
                mCameraAngle;
  
  // half-angle (degrees) of the view through the camera
  private float mViewWidth;
  
  // scenery objects
  private Mountains mMountains;
  private CloudySky mSky;

  // game objects
  private LinkedList<Sprite> mSprites;
  private Player             mPlayer;
          
  // constructor
  public Game() {

    // set up lights
    final int numLights = kLightElevations.length;
    mLightDirections = new float[numLights][4];
    for ( int k = 0 ; k < numLights ; k++ ) {
      float phi   = kLightElevations[k];
      float r     = (float)Math.cos(phi*Math.PI/180.0),
            y     = (float)Math.sin(phi*Math.PI/180.0);
      float theta = kLightFirstAngle + (360.0f*k)/numLights;
      float x     = r*(float)Math.cos(theta*Math.PI/180.0),
            z     = r*(float)Math.sin(theta*Math.PI/180.0);
      mLightDirections[k][0] = x;
      mLightDirections[k][1] = y;
      mLightDirections[k][2] = z;
      mLightDirections[k][3] = 0.0f;
    }

    // set up scenery objects
    mMountains = new Mountains();
    mSky = new CloudySky();

    // set up game objects
    mPlayer = new Player();
    mSprites = new LinkedList<Sprite>();
    mSprites.add(mPlayer);
    mSprites.add(new EnemyTank(0.0f));
    mSprites.add(new EnemyTank(120.0f));
    mSprites.add(new EnemyTank(240.0f));
    mSprites.add(new Obstacle(0.0f, 0.0f, 20.0f));
    mSprites.add(new Obstacle(-20.0f, +20.0f, 10.0f));
    mSprites.add(new Obstacle(-20.0f, -20.0f, 30.0f));
    mSprites.add(new Obstacle(+20.0f, -20.0f, 40.0f));
    mSprites.add(new Obstacle(+20.0f, +20.0f, 50.0f));

    // initial camera
    positionCamera();

    mOverviewMode   = false;
    mOverviewToggle = true;
    mOverviewScale  = kOverviewScaleInitial;
    
  } // constructor
  
  // initialize
  public void init(GL gl) {
    
    gl.glClearColor(kBackgroundColour[0], 
                    kBackgroundColour[1],
                    kBackgroundColour[2], 1.0f);
    
    gl.glEnable(GL.GL_CULL_FACE);
    gl.glEnable(GL.GL_DEPTH_TEST);

    gl.glEnable(GL.GL_LIGHTING);
    gl.glEnable(GL.GL_LIGHT0);
    gl.glLightfv(GL.GL_LIGHT0, GL.GL_DIFFUSE, kLightDiffuse, 0);
    gl.glLightfv(GL.GL_LIGHT0, GL.GL_SPECULAR, kLightSpecular, 0);
    gl.glEnable(GL.GL_LIGHT1);
    gl.glLightfv(GL.GL_LIGHT1, GL.GL_DIFFUSE, kLightDiffuse, 0);
    gl.glLightfv(GL.GL_LIGHT1, GL.GL_SPECULAR, kLightSpecular, 0);
    gl.glEnable(GL.GL_LIGHT2);
    gl.glLightfv(GL.GL_LIGHT2, GL.GL_DIFFUSE, kLightDiffuse, 0);
    gl.glLightfv(GL.GL_LIGHT2, GL.GL_SPECULAR, kLightSpecular, 0);
    
    gl.glBlendFunc(GL.GL_SRC_ALPHA, GL.GL_ONE_MINUS_SRC_ALPHA);
    gl.glLineWidth(kLineWidth);
    
    gl.glEnable(GL.GL_NORMALIZE);

    Obstacle.init(gl);
    Tank.init(gl);
    
  } // init()

  // set up the camera projection
  // (always called in GL_PROJECTION matrix mode)
  public void configureCamera(GL gl) {
  
    float w = kCameraLensScale*Env.aspectRatio(),
          h = kCameraLensScale;
    gl.glFrustum(-w, +w, -h, +h, kCameraNearPlane, kCameraFarPlane);
    
  } // configureCamera()
  
  // set up the camera's position and direction
  // (slightly behind the player, pointing in the same direction)
  private void positionCamera() {
  
    mCameraAngle = mPlayer.getAngle();

    double theta = mCameraAngle*Math.PI/180.0;
    mCameraX = mPlayer.getXPos() - kCameraOffset*(float)Math.cos(theta);
    mCameraZ = mPlayer.getZPos() - kCameraOffset*(float)Math.sin(theta);

  } // positionCamera()
  
  // update positions of objects
  public void advance(float dt) {
  
    updateOverview(dt);
  
    mPlayer.advance(dt);
    positionCamera();
  
    mSky.advance(dt);

    for ( Sprite s : mSprites ) {
      if ( s instanceof Player ) continue;
      s.advance(dt);
    }
    
  } // advance()
  
  // display
  public void display(GL gl) {
  
    float v = Env.aspectRatio() * kCameraLensScale / kCameraNearPlane;
    mViewWidth = (float)( (180.0f/Math.PI) * Math.atan(v) );
    
    if ( mOverviewMode ) displayOverview(gl);
    else                 displayNormal(gl);
    
  } // display()
  
  // display from first-person perspective
  private void displayNormal(GL gl) {
    
    gl.glClear(GL.GL_DEPTH_BUFFER_BIT);

    // camera
    gl.glPushMatrix();
    gl.glRotatef(mCameraAngle+90.0f, 0.0f, 1.0f, 0.0f);
    gl.glTranslatef(-mCameraX, -kCameraY, -mCameraZ);

    // mountains and sky
    enterOrthoMode(gl);
    mSky.display(gl, mCameraAngle, mViewWidth);
    mMountains.display(gl, mCameraAngle, mViewWidth);
    exitOrthoMode(gl);
    
    // objects (reflected)
    gl.glPushMatrix();
    gl.glScalef(1.0f, -1.0f, 1.0f);
    gl.glFrontFace(GL.GL_CW);
    displayObjects(gl);
    gl.glFrontFace(GL.GL_CCW);
    gl.glPopMatrix();

    // reflective surface on the lower half of the screen
    displaySurface(gl);
                
    // objects (unreflected)
    gl.glClear(GL.GL_DEPTH_BUFFER_BIT);
    displayObjects(gl);
    
    gl.glPopMatrix();
    
  } // displayNormal()

  // display from above
  private void displayOverview(GL gl) {

    gl.glClear(GL.GL_COLOR_BUFFER_BIT | GL.GL_DEPTH_BUFFER_BIT);

    gl.glMatrixMode(GL.GL_PROJECTION);
    gl.glPushMatrix();
    gl.glLoadIdentity();
    float xScale = mOverviewScale * Env.aspectRatio(),
          yScale = mOverviewScale,
          height = 20.0f;
    gl.glOrtho(-xScale, +xScale, -yScale, +yScale, -height, +height);
    gl.glMatrixMode(GL.GL_MODELVIEW);
    gl.glPushMatrix();
    gl.glLoadIdentity();
    gl.glRotatef(+90.0f, 1.0f, 0.0f, 0.0f);
    gl.glTranslatef(-mPlayer.getXPos(), 0.0f, -mPlayer.getZPos());

    gl.glDisable(GL.GL_LIGHT0);
    gl.glDisable(GL.GL_LIGHT1);
    gl.glDisable(GL.GL_LIGHT2);
      
    displayViewArea(gl);
      
    for ( Sprite s : mSprites ) s.display(gl);

    gl.glEnable(GL.GL_LIGHT0);
    gl.glEnable(GL.GL_LIGHT1);
    gl.glEnable(GL.GL_LIGHT2);
      
    gl.glPopMatrix();
    gl.glMatrixMode(GL.GL_PROJECTION);
    gl.glPopMatrix();
    gl.glMatrixMode(GL.GL_MODELVIEW);
    
  } // displayOverview()
    
  // display all objects (without reflection)
  private void displayObjects(GL gl) {

    gl.glLightfv(GL.GL_LIGHT0, GL.GL_POSITION, mLightDirections[0], 0);
    gl.glLightfv(GL.GL_LIGHT1, GL.GL_POSITION, mLightDirections[1], 0);
    gl.glLightfv(GL.GL_LIGHT2, GL.GL_POSITION, mLightDirections[2], 0);

    // display all objects except the player
    for ( Sprite s : mSprites ) {
      if ( s instanceof Player ) continue;
      s.display(gl);
    }
            
  } // displayObjects()

  // render the ground
  private void displaySurface(GL gl) {

    assert( !mOverviewMode );
  
    enterOrthoMode(gl);
  
    gl.glEnable(GL.GL_BLEND);
    gl.glShadeModel(GL.GL_SMOOTH);

    gl.glBegin(GL.GL_QUAD_STRIP);
      gl.glColor4fv(kSurfaceColourNear, 0);
      gl.glVertex2f(-Env.aspectRatio(), -1.0f);
      gl.glVertex2f(+Env.aspectRatio(), -1.0f);
      gl.glVertex2f(-Env.aspectRatio(), kSurfaceNearMark);
      gl.glVertex2f(+Env.aspectRatio(), kSurfaceNearMark);
      gl.glColor4fv(kSurfaceColourFar, 0);
      gl.glVertex2f(-Env.aspectRatio(), kSurfaceFarMark);
      gl.glVertex2f(+Env.aspectRatio(), kSurfaceFarMark);
      gl.glVertex2f(-Env.aspectRatio(), 0.0f);
      gl.glVertex2f(+Env.aspectRatio(), 0.0f);
    gl.glEnd();
    
    gl.glDisable(GL.GL_BLEND);
 
    exitOrthoMode(gl);
    
  } // displaySurface()

  // shows in overview mode the visible area in first-person mode
  private void displayViewArea(GL gl) {
  
    assert( mOverviewMode );
    
    gl.glDisable(GL.GL_LIGHTING);
    gl.glEnable(GL.GL_BLEND);
    gl.glEnable(GL.GL_POLYGON_SMOOTH);
    gl.glShadeModel(GL.GL_FLAT);
    
    float dNear = Env.aspectRatio()*kCameraLensScale,
          dFar  = dNear*kCameraFarPlane/kCameraNearPlane;
    float x0 = mCameraX,
          z0 = mCameraZ;
    float xv = (float)Math.cos(mCameraAngle*Math.PI/180.0),
          zv = (float)Math.sin(mCameraAngle*Math.PI/180.0);
    float dx = -zv,
          dz = +xv;
    float x1 = x0 + kCameraNearPlane*xv + dNear*dx,
          z1 = z0 + kCameraNearPlane*zv + dNear*dz;
    float x2 = x0 + kCameraFarPlane*xv  + dFar*dx,
          z2 = z0 + kCameraFarPlane*zv  + dFar*dz;
    float x3 = x0 + kCameraFarPlane*xv  - dFar*dx,
          z3 = z0 + kCameraFarPlane*zv  - dFar*dz;
    float x4 = x0 + kCameraNearPlane*xv - dNear*dx,
          z4 = z0 + kCameraNearPlane*zv - dNear*dz;
    
    gl.glColor4fv(kOverviewColour, 0);
    gl.glBegin(GL.GL_QUADS);
      gl.glVertex3f(x1, -0.1f, z1);
      gl.glVertex3f(x2, -0.1f, z2);
      gl.glVertex3f(x3, -0.1f, z3);
      gl.glVertex3f(x4, -0.1f, z4);
    gl.glEnd();
  
    gl.glDisable(GL.GL_POLYGON_SMOOTH);
    gl.glDisable(GL.GL_BLEND);
    gl.glEnable(GL.GL_LIGHTING);
    
  } // displayViewArea()
  
  // switch to orthographic mode (2D coords -aspect<x<+aspect, -1<y<+1)
  private void enterOrthoMode(GL gl) {
  
    gl.glMatrixMode(GL.GL_PROJECTION);
    gl.glPushMatrix();
    gl.glLoadIdentity();
    Env.glu().gluOrtho2D(-Env.aspectRatio(), +Env.aspectRatio(), -1.0f, 1.0f);
    
    gl.glMatrixMode(GL.GL_MODELVIEW);
    gl.glPushMatrix();
    gl.glLoadIdentity();

    gl.glDisable(GL.GL_DEPTH_TEST);
    gl.glDisable(GL.GL_LIGHTING);

  } // enterOrthoMode()

  // switch out of orthographic mode
  private void exitOrthoMode(GL gl) {
  
    gl.glPopMatrix();
    gl.glMatrixMode(GL.GL_PROJECTION);
    gl.glPopMatrix();
    gl.glMatrixMode(GL.GL_MODELVIEW);
 
    gl.glEnable(GL.GL_DEPTH_TEST);
    gl.glEnable(GL.GL_LIGHTING);

  } // exitOrthoMode()

  // toggle overview mode if necessary
  private void updateOverview(float dt) {
  
    final boolean keyMode  = Env.keys().pressed(KeyEvent.VK_1),
                  keySmall = Env.keys().pressed(KeyEvent.VK_2),
                  keyBig   = Env.keys().pressed(KeyEvent.VK_3);
                  
    if ( !mOverviewToggle ) {
      if ( !keyMode ) mOverviewToggle = true;
    } else if ( mOverviewMode ) {
      if ( keyMode ) {
        mOverviewMode = false;
        mOverviewToggle = false;
      }
      if ( keySmall ) {
        mOverviewScale *= (float)Math.exp(-kOverviewScaleRate*dt);
        mOverviewScale = Math.max(kOverviewScaleMin, mOverviewScale);
      }
      if ( keyBig ) {
        mOverviewScale *= (float)Math.exp(+kOverviewScaleRate*dt);
        mOverviewScale = Math.min(kOverviewScaleMax, mOverviewScale);
      }
    } else if ( keyMode ) {
      mOverviewMode = true;
      mOverviewToggle = false;
    }
    
  } // updateOverview()
  
} // class Game
 