/*
 *  CloudySky.java
 */

package Experiment;

import java.nio.*;
import javax.media.opengl.*;

// animated clouds (a 2D texture drawn in the background)
public class CloudySky {

  // number of points (x and y) in pattern
  static private final int kSize = 64;

  // noisiness of the noise
  static private final float kNoisePersist = 0.5f;
  
  // "height" values below which foreground gives way to background (0 to 100)
  static private final int kBackgroundCutoff = 30,
                           kForegroundCutoff = 70;
  
  // direction to light source
  static private final float kLightAngle = -0.6f*(float)Math.PI,
                             kLightDx    = (float)Math.cos(kLightAngle),
                             kLightDy    = (float)Math.sin(kLightAngle);
  
  // limits on gradient corresponding to light and dark
  static private final float kDarkGradient  = -0.10f,
                             kLightGradient = +0.15f;
  
  // colours defined as RGB
  static private final float[] kColourDark  = { 0.70f, 0.70f, 0.70f },
                               kColourLight = { 1.00f, 0.50f, 0.50f},
                               kColourSky   = { 0.30f, 0.60f, 0.90f };
  
  // range of speeds that images move at within the tile
  static private final float kMinSpeed = 6.0f,
                             kMaxSpeed = 12.0f;
  
  // number of times sky image is repeated to cover 360 degrees
  static private final int kNumRepeats = 6;
  
  // two tile-size arrays of height values between 0 and 100
  private byte[][] mHeightsA = null,
                   mHeightsB = null;
  
  // two tile-size arrays of shade values corresponding to height arrays above
  // (shade values are scaled from -100 to +100)
  private byte[][] mShadesA = null,
                   mShadesB = null;
  
  // the sky image stored as a buffer of (unsigned) RGB bytes
  private ByteBuffer mColours = null;

  // identifier of the JOGL texture object (0 if not known yet)
  private int mTextureName;
  
  // image drift: position offset and speed
  private float mXOffsetB,
                mXSpeedB;
  
  // constructor
  public CloudySky() {

    PerlinNoise noise = new PerlinNoise(2*kSize, kSize, kNoisePersist);
    mHeightsA = convertNoiseToHeights(noise);
    mShadesA = convertNoiseToShades(noise);
    
    noise = new PerlinNoise(2*kSize, kSize, kNoisePersist);
    mHeightsB = convertNoiseToHeights(noise);
    mShadesB = convertNoiseToShades(noise);
    
    mXOffsetB = 0.0f;
    mXSpeedB  = (Env.randomBoolean() ? +1 : -1)
                *Env.randomFloat(kMinSpeed, kMaxSpeed);
    
    mColours = ByteBuffer.allocateDirect(3*kSize*kSize);
    mColours.order(ByteOrder.nativeOrder());

    mTextureName = 0;
    
    constructImage();
    
  } // constructor
  
  // animate the image
  public void advance(float dt) {
    
    mXOffsetB = Env.fold(mXOffsetB + dt*mXSpeedB, kSize);
    constructImage();
    
  } // advance()
  
  // scale noise values between 0 and 1 to integer heights between 0 and 100
  private byte[][] convertNoiseToHeights(PerlinNoise noise) {

    byte[][] heights = new byte[kSize][kSize];
    
    for ( int i = 0 ; i < kSize ; i++ ) {
      for ( int j = 0 ; j < kSize ; j++ ) {
        heights[i][j] = (byte)(100.0f*noise.mData[2*i][j]);
      }
    }
    
    return heights;
    
  } // convertNoiseToHeights()
  
  // for a tile of height values between 0 and 1 
  // return shade values such that -100 is dark and +100 is light
  // (lighting is based simply on gradient, not proper light source)
  private byte[][] convertNoiseToShades(PerlinNoise noise) {
    
    byte[][] shades = new byte[kSize][kSize];

    for ( int i = 0 ; i < kSize ; i++ ) {
      for ( int j = 0 ; j < kSize ; j++ ) {
        
        // partial derivatives of height function
        float dx = ( noise.mData[2*Env.fold(i+1,kSize)][j]
                   - noise.mData[2*Env.fold(i-1,kSize)][j] )/2.0f,
              dy = ( noise.mData[2*i][Env.fold(j+1,kSize)]
                   - noise.mData[2*i][Env.fold(j-1,kSize)] )/2.0f;
        
        float // gradient along light direction (<0 for dark, >0 for light)
              lightGradient = -(dx*kLightDx + dy*kLightDy),
              // scale such that kDarkGradient -> -1, kLightGradient -> +1
              lightScaled = 2.0f*(lightGradient-kDarkGradient)
                                /(kLightGradient-kDarkGradient) - 1.0f,
              // clip to this range
              lightClipped = Math.min(+1.0f, Math.max(-1.0f, lightScaled)); 

        // fill in array
        shades[i][j] = (byte)(100.0f*lightClipped);
        
      }
    }

    return shades;
    
  } // convertNoiseToShades()
  
  // paint the tile's image
  // (integer arithmetic to help performance)
  private void constructImage() {
    
    final short rBase  = (short)(255*kColourDark[0]),
                gBase  = (short)(255*kColourDark[1]),
                bBase  = (short)(255*kColourDark[2]);
    final short rDelta = (short)(255*(kColourLight[0] - kColourDark[0])),
                gDelta = (short)(255*(kColourLight[1] - kColourDark[1])),
                bDelta = (short)(255*(kColourLight[2] - kColourDark[2]));
    final short rSky   = (short)(255*kColourSky[0]),
                gSky   = (short)(255*kColourSky[1]),
                bSky   = (short)(255*kColourSky[2]);
    
    mColours.clear();
    
    for ( int i = 0 ; i < kSize ; i++ ) {
      int jB1 = Env.fold((int)Math.floor(mXOffsetB), kSize),
          jB2 = Env.fold(jB1+1, kSize);
      int dx = (int)Math.floor(256*(mXOffsetB - Math.floor(mXOffsetB)));
      for ( int jA = 0 ; jA < kSize ; jA++ ) {

        int heightB = ( (256-dx)*(int)mHeightsB[i][jB1] 
                      + dx*(int)mHeightsB[i][jB2] )/256;
        int height  = (int)mHeightsA[i][jA] + heightB - 50;
        
        short r = rSky,
              g = gSky, 
              b = bSky;
        if ( height >= kBackgroundCutoff ) {
          int shadeB = ( (256-dx)*(int)mShadesB[i][jB1] 
                       + dx*(int)mShadesB[i][jB2] )/256;
          int shade = (int)mShadesA[i][jA] + shadeB;
          int frac  = Math.min(+100, Math.max(-100, shade)) + 100;
          r = (short)(rBase + (rDelta*frac)/200);
          g = (short)(gBase + (gDelta*frac)/200);
          b = (short)(bBase + (bDelta*frac)/200);
          if ( height <= kForegroundCutoff ) {
            int frac2 = (100*(height - kBackgroundCutoff))
                        /(kForegroundCutoff - kBackgroundCutoff);
            r = (short)(rSky + ((r-rSky)*frac2)/100);
            g = (short)(gSky + ((g-gSky)*frac2)/100);
            b = (short)(bSky + ((b-bSky)*frac2)/100);
          }
        } 
        
        mColours.put((byte)r);
        mColours.put((byte)g);
        mColours.put((byte)b);
        
        if ( ++jB1 == kSize ) jB1 = 0;
        if ( ++jB2 == kSize ) jB2 = 0;
      } // down tile
    } // across tile

    mColours.flip();
    
  } // constructImage()
  
  // turn the image into a texture
  // (note: the function also binds the texture)
  private void constructTexture(GL gl) {

    gl.glPixelStorei(GL.GL_UNPACK_ALIGNMENT, 1);

    if ( mTextureName == 0 ) {
    
      // first time through the texture is created
      int names[] = new int[1];
      gl.glGenTextures(1, names, 0);
      mTextureName = names[0];
      
      gl.glBindTexture(GL.GL_TEXTURE_2D, mTextureName);
    
      gl.glTexParameteri(GL.GL_TEXTURE_2D, 
                         GL.GL_TEXTURE_WRAP_S, GL.GL_REPEAT);
      gl.glTexParameteri(GL.GL_TEXTURE_2D, 
                         GL.GL_TEXTURE_WRAP_T, GL.GL_CLAMP);
      gl.glTexParameteri(GL.GL_TEXTURE_2D, 
                         GL.GL_TEXTURE_MAG_FILTER, GL.GL_LINEAR);
      gl.glTexParameteri(GL.GL_TEXTURE_2D, 
                         GL.GL_TEXTURE_MIN_FILTER, GL.GL_NEAREST);
    
      gl.glTexImage2D(GL.GL_TEXTURE_2D, 0, GL.GL_RGB,
                      kSize, kSize, 0, 
                      GL.GL_RGB, GL.GL_UNSIGNED_BYTE, mColours);
      
    } else {
    
      // redefine the existing texture object
      gl.glBindTexture(GL.GL_TEXTURE_2D, mTextureName);
    
      gl.glTexSubImage2D(GL.GL_TEXTURE_2D, 0, 
                         0, 0, kSize, kSize,
                         GL.GL_RGB, GL.GL_UNSIGNED_BYTE, mColours);
      
    }
    
  } // constructTexture()
  
  // display the sky (together with its reflection)
  // ("viewAngle" is the camera direction in degrees)
  // ("viewWidth" is the half-angle of the view)
  // (game expected to be in 'ortho' mode, |x|<=AspectRatio & |y|<=1)
  // (note: always assumed that the image has changed since the last call)
  public void display(GL gl, float viewAngle, float viewWidth) {
  
    constructTexture(gl);
    
    gl.glShadeModel(GL.GL_FLAT);
    gl.glEnable(GL.GL_TEXTURE_2D);
    gl.glTexEnvf(GL.GL_TEXTURE_ENV, GL.GL_TEXTURE_ENV_MODE, GL.GL_REPLACE);
    
    float theta0 = viewAngle - viewWidth, 
          dtheta = 360.0f/kNumRepeats,
          s0     = Env.fold(theta0/dtheta, 1.0f),
          s1     = s0 + 2*viewWidth/dtheta;
    
    gl.glBegin(GL.GL_QUADS);

      // upper half screen (real sky)
      gl.glTexCoord2f(s0, 0.0f);
      gl.glVertex2f(-Env.aspectRatio(), 0.0f);
      gl.glTexCoord2f(s1, 0.0f);
      gl.glVertex2f(+Env.aspectRatio(), 0.0f);
      gl.glTexCoord2f(s1, 1.0f);
      gl.glVertex2f(+Env.aspectRatio(), +1.0f);
      gl.glTexCoord2f(s0, 1.0f);
      gl.glVertex2f(-Env.aspectRatio(), +1.0f);
      
      // lower half screen (reflected sky)
      gl.glTexCoord2f(s0, 0.0f);
      gl.glVertex2f(-Env.aspectRatio(), 0.0f);
      gl.glTexCoord2f(s0, 1.0f);
      gl.glVertex2f(-Env.aspectRatio(), -1.0f);
      gl.glTexCoord2f(s1, 1.0f);
      gl.glVertex2f(+Env.aspectRatio(), -1.0f);
      gl.glTexCoord2f(s1, 0.0f);
      gl.glVertex2f(+Env.aspectRatio(), 0.0f);
      
    gl.glEnd();    
    
    gl.glDisable(GL.GL_TEXTURE_2D);
 
  } // display()
    
} // class CloudySky
